<?php

##################################################
#
# Copyright (c) 2004-2006 OIC Group, Inc.
# Written and Designed by James Hunt
#
# This file is part of Exponent
#
# Exponent is free software; you can redistribute
# it and/or modify it under the terms of the GNU
# General Public License as published by the Free
# Software Foundation; either version 2 of the
# License, or (at your option) any later version.
#
# GPL: http://www.gnu.org/licenses/gpl.txt
#
##################################################

/* exdoc
 * The definition of this constant lets other parts
 * of the system know that the DateTime Subsystem
 * has been included for use.
 * @node Subsystems:DateTime
 */
define("SYS_DATETIME",1);

/* exdoc
 * @state <b>UNDOCUMENTED</b>
 * @node To Be Deprecated
 */
function exponent_datetime_monthsDropdown($controlName,$default_month) {
	$months = array(
		1=>"January",
		2=>"February",
		3=>"March",
		4=>"April",
		5=>"May",
		6=>"June",
		7=>"July",
		8=>"August",
		9=>"September",
		10=>"October",
		11=>"November",
		12=>"December"
	);

	$html = '<select name="' . $controlName . '" size="1">';
	foreach ($months as $id=>$month) {
		$html .= '<option value="' . $id . '"';
		if ($id == $default_month) $html .= ' selected';
		$html .= '>' . $month . '</option>';
	}
	$html .= '</select>';
	return $html;
}

/* exdoc
 * Looks at a start and end time and figures out
 * how many seconds elapsed since between the earlier
 * timestamp and the later timestamp.  It doesn't matter
 * if the bigger argument is specified first or not. Returns
 * the number of seconds between $time_a and $time_b
 *
 * @param timestamp $time_a The first timestamp
 * @param timestamp $time_b The second timestamp
 * @node Subsystems:DateTime
 */
function exponent_datetime_duration($time_a,$time_b) {
	$d = abs($time_b-$time_a);
	$duration = array();
	if ($d >= 86400) {
		$duration['days'] = floor($d / 86400);
		$d %= 86400;
	}
	if (isset($duration['days']) || $d >= 3600) {
		if ($d) $duration['hours'] = floor($d / 3600);
		else $duration['hours'] = 0;
		$d %= 3600;
	}
	if (isset($duration['hours']) || $d >= 60) {
		if ($d) $duration['minutes'] = floor($d / 60);
		else $duration['minutes'] = 0;
		$d %= 60;
	}
	$duration['seconds'] = $d;
	return $duration;
}

/* exdoc
 * Given a timestamp, this function will calculate another timestamp
 * that represents the beginning of the month that the passed timestamp
 * falls into.  For instance, passing a timestamp representing January 25th 1984
 * would return a timestamp representing January 1st 1984, at 12:00am.
 *
 * @param timestamp $timestamp The original timestamp to use when calculating.
 * @node Subsystems:DateTime
 */
function exponent_datetime_startOfMonthTimestamp($timestamp) {
	$info = getdate($timestamp);
	// Calculate the timestamp at 8am, and then subtract 8 hours, for Daylight Savings
	// Time.  If we are in those strange edge cases of DST, 12:00am can turn out to be
	// of the previous day.
	return mktime(0,0,0,$info['mon'],1,$info['year']);
}

/* exdoc
 * Given a timestamp, this function will calculate another timestamp
 * that represents the end of the month that the passed timestamp
 * falls into.  For instance, passing a timestamp representing January 25th 1984
 * would return a timestamp representing January 31st 1984, at 11:59pm.
 *
 * @param timestamp $timestamp The original timestamp to use when calculating.
 * @node Subsystems:DateTime
 */
function exponent_datetime_endOfMonthTimestamp($timestamp) {
	$info = getdate($timestamp);
	// No month has fewer than 28 days, even in leap year, so start out at 28.
	// At most, we will loop through the while loop 3 times (29th, 30th, 31st)
	$info['mday'] = 28;
	// Keep incrementing the mday value until it is not valid, and use last valid value.
	// This should get us the last day in the month, and take into account leap years
	while (checkdate($info['mon'],$info['mday']+1,$info['year'])) $info['mday']++;
	// Calculate the timestamp at 8am, and then subtract 8 hours, for Daylight Savings
	// Time.  If we are in those strange edge cases of DST, 12:00am can turn out to be
	// of the previous day.
	return mktime(0,0,0,$info['mon'],$info['mday'],$info['year']);
}

/* exdoc
 * Looks at a timestamp and returns the date of the last
 * day.  For instance, if the passed timestamp was in January,
 * this function would return 31.  Leap year is taken into account.
 *
 * @param timestamp $timestamp The timestamp to check.
 * @node Subsystems:DateTime
 */
function exponent_datetime_endOfMonthDay($timestamp) {
	$info = getdate($timestamp);
	// No month has fewer than 28 days, even in leap year, so start out at 28.
	// At most, we will loop through the while loop 3 times (29th, 30th, 31st)
	$last = 28;
	// Keep incrementing the mday value until it is not valid, and use last valid value.
	// This should get us the last day in the month, and take into account leap years
	while (checkdate($info['mon'],$last+1,$info['year'])) $last++;
	return $last;
}

/* exdoc
 * Looks at a timestamp and returns another timestamp representing
 * 12:00:01 am of the same day.
 *
 * @param timestamp $timestamp The timestamp to check.
 * @node Subsystems:DateTime
 */
function exponent_datetime_startOfDayTimestamp($timestamp) {
	$info = getdate($timestamp);
	// Calculate the timestamp at 8am, and then subtract 8 hours, for Daylight Savings
	// Time.  If we are in those strange edge cases of DST, 12:00am can turn out to be
	// of the previous day.
	return mktime(0,0,0,$info['mon'],$info['mday'],$info['year']);
}
/* exdoc
 * Looks at a timestamp and returns another timestamp representing
 * 12:00:01 am of the Sunday of the same week.
 *
 * @param timestamp $timestamp The timestamp to check.
 * @node Subsystems:DateTime
 */
function exponent_datetime_startOfWeekTimestamp($timestamp) {
	$info = getdate($timestamp);

    // 0 (for Sunday) through 6 (for Saturday)
    // on Sunday, if week starts on Monday, get last week through today.
    $weekday = $info['wday'];
    if ( ($weekday == 0) && (DISPLAY_START_OF_WEEK == 1) ){
        $weekday = 7;
    }

    // calculate day offset to start of week
    $days_to_week_start = 0 - $weekday + DISPLAY_START_OF_WEEK;

	$week_start = mktime(12, 0, 0, date("m",$timestamp), date("d",$timestamp) + $days_to_week_start, date("Y",$timestamp));

	return $week_start;
}

// Recurring Dates

/* exdoc
 * Find all of the dates that fall within the daily recurrence criteria.
 * This is the simplest form of recurrence, events are spaced a given
 * number of days apart.
 *
 * @param timestamp $start The start of the recurrence range
 * @param timestamp $end The end of the recurrence range
 * @param integer $freq Frequency of recurrence - 2 means every 2 days, and 1
 * 	means every day.
 * @node Subsystems:DateTime
 */
function exponent_datetime_recurringDailyDates($start,$end,$freq) {
	$dates = array();
	$curdate = $start;
	do {
		$dates[] = $curdate;
		$curdate = exponent_datetime_startOfDayTimestamp($curdate + ((86400) * $freq)+3601);
	} while ($curdate <= $end);
	return $dates;
}

/* exdoc
 * Finds all of the dates that fall within the weekly recurrence criteria
 * (namely, which weekdays) and within the $start to $end timestamp range.
 *
 * For a technical discussion of this function and the mathematics involved,
 * please see the sdk/analysis/subsystems/datetime.txt file.
 *
 * @param timestamp $start The start of the recurrence range
 * @param timestamp $end The end of the recurrence range
 * @param integer $freq Weekly frequency - 1 means every week, 2 means every
 *   other week, etc.
 * @param array $days The weekdays (in integer notation, 0 = Sunday, etc.) that
 *   should be matched.  A MWF recurrence rotation would contain the values
 *  1,3 and 5.
 * @node Subsystems:DateTime
 */
function exponent_datetime_recurringWeeklyDates($start,$end,$freq,$days) {
	// Holding array, for keeping the timestamps of applicable dates.
	// This variable will be returned to the calling scope.
	$dates = array();

	// Need to figure out which weekday occurs directly after the specified
	// start date.  This will be our launching point for the recurrence calculations.
	$dateinfo = getdate($start);

	// Finding the Start Date
	//
	// Start at the first weekday in the list ($days[$counter] where $counter is 0)
	// and go until we find a weekday greater than the weekday of the $start date.
	//
	// So, if we start on a Tuesday, and want to recur weekly for a MWF rotation,
	// This would check Monday, then Wednesday and stop, using wednesday for the
	// recacluated start date ($curdate)
	for ($counter = 0; $counter < count($days); $counter++) {
		if ($days[$counter] >= $dateinfo['wday']) {
			// exit loop, we found the weekday to use ($days[$counter])
			break;
		}
	}
	if ($days[$counter] < $dateinfo['wday']) {
		// in case we did MWF and started on a Saturday...
		$counter = 0; // reset to first day of next week
		$start += 86400 * (7-$dateinfo['wday']+$days[$counter]);
	} else if ($days[$counter] > $dateinfo['wday']) {
		// 'Normal' case, in which we started before one of the repeat days.
		$start += 86400 * ($days[$counter] - $dateinfo['wday']);
	}
	// Found start date.  Set curdate to the start date, so it gets picked
	// up in the do ... while loop.
	$curdate = $start;

	// Find all of the dates that match the recurrence criteria, within the
	// specified recurrence range.  Implemented as a do ... while loop because
	// we always need at least the start date, and we have already determined
	// that with the code above (the $curdate variable)
	do {
		// Append $curdate to the array of dates.  $curdate will be changed
		// at the end of the loop, to be equal to the next date meeting the
		// criteria.  If $curdate ever falls outside the recurrence range, the
		// loop will exit.
		$dates[] = $curdate;
		$curdate += 8*3600; // Add 8 hours, to avoid DST problems
		// Grab the date information (weekday, month, day, year, etc.) for
		// the current date, so we can ratchet up to the next date in the series.
		$dateinfo = getdate($curdate);
		// Get the current weekday.
		$day = $days[$counter];
		// Increment the counter so that the next time through we get the next
		// weekday.  If the counter moves off the end of the list, reset it to 0.
		$counter++;
		if ($counter >= count($days)) {
			// Went off the end of the week.  Reset the pointer to the beginning
			$counter = 0;
			// Difference in number of days between the last day in the rotation
			// and the first day (for recacluating the $curdate value)
			#$daydiff = $days[count($days)-1]-$days[0];

			$daydiff = 7 + $days[0] - $days[count($days)-1];

			if ($daydiff == 0) {
				// In case we are in a single day rotation, the difference will be 0.
				// It needs to be 7, so that we skip ahead a full week.
				$daydiff = 7;
			}
			// Increment the current date to go off to the next week, first weekday
			// in the rotation.
			$curdate += 7 * 86400 * ($freq-1) + 86400 * $daydiff; // Increment by number of weeks
		} else {
			// If we haven't gone off the end of the week, we just need to add the number
			// of days between the next weekday in the rotation ($days[$counter] - because
			// $counter was incremented above) and the $curdate weekday (store in the
			// $dateinfo array returned from the PHP call to getdate(), aboce).
			$curdate += 86400 * ($days[$counter] - $dateinfo['wday']);
		}
		// Round down to the start of the day (12:00 am) for $curdate, in case something
		// got a little out of whack thanks to DST.
		$curdate = exponent_datetime_startOfDayTimestamp($curdate);
	} while ($curdate <= $end); // If we go off the end of the recurrence range, ext.

	// We have no fully calculated the dates we need. Return them to the calling scope.
	return $dates;
}

/* exdoc
 * Finds all of the dates that fall within the monthly recurrence criteria
 * and within the $start to $end timestamp range.  Monthly recurrence can be
 * done on a specific date (the 14th of the month) or on a specific weekday / offset
 * pair (the third sunday of the month).
 *
 * @param timestamp $start The start of the recurrence range
 * @param timestamp $end The end of the recurrence range
 * @param integer $freq Monthly frequency - 1 means every month, 2 means every
 *   other month, etc.
 * @param bool $by_day Whether or not to recur by the weekday and week offset
 * (in case of true), or by the date (in case of false).
 * @node Subsystems:DateTime
 */
function exponent_datetime_recurringMonthlyDates($start,$end,$freq,$by_day=false) {
	// Holding array, for keeping all of the matching timestamps
	$dates = array();
	// Date to start on.
	$curdate = $start;

	// Get the date info, including the weekday.
	$dateinfo = getdate($curdate);

	// Store the month day.  If we are not doing by day monthly recurrence,
	// then this will be used unchanged throughout the do .. while loop.
	$mdate = $dateinfo['mday'];

	$week = 0; // Only used for $by_day;
	$wday = 0; // Only used for $by_day;
	if ($by_day) {
		// For by day recurrence, we need to know what week it is, and what weekday.
		// (i.e. the 3rd Thursday of the month)

		// Calculate the Week Offset, as the ceilling value of date / DAYS_PER_WEEK
		$week = ceil($mdate / 7);
		// Store the weekday
		$wday = $dateinfo['wday'];
	}

//	eDebug($dateinfo);
	// Loop until we exceed the until date.
	do {
		// Append the current date to the list of dates.  $curdate will be updated
		// in the rest of the loop, so that it contains the next date.  This next date will
		// be checked in the while condition, and if it is still before the until date,
		// the loop iterates back here again for another go.
		$dates[] = $curdate;

		// Grab the date information for $curdate.  This gives us the current month
		// information, for the next jump.
		//$dateinfo = getdate($curdate);

		// Make the next month's timestamp, by adding frequency to the month.
		// PHP can pick up on the fact that the 13th month of this year is the 1st
		// month of the next year.
		$curdate = mktime(8,0,0,$dateinfo['mon']+$freq,1,$dateinfo['year']);
		$dateinfo = getdate($curdate);
		//eDebug($dateinfo);

		// Manually update the month and monthday.
		//eDebug($dateinfo);
		//$dateinfo['mon'] += $freq;  	//Bump the month to next month
		//eDebug($freq);
		//eDebug($dateinfo);
		//$dateinfo['mday'] = 1;		//Set the day of the month to the first.
		//eDebug($dateinfo);
		//exit();

		if ($by_day) {
			if ($dateinfo['wday'] > $wday) {
				$mdate = $wday - $dateinfo['wday'] + ( 7 * $week ) + 1;
				//echo "month day: ".$mdate."<br>";
			} elseif ($dateinfo['wday'] <= $wday) {
				$mdate = $wday - $dateinfo['wday'] + ( 7 * ( $week - 1 ) ) + 1;
				//echo "month day: ".$mdate."<br>";
			}

			// For by day recurrence (first tuesday of every month), we need to do a
			// little more fancy footwork to determine the next timestamp, since there
			// is no easy mathematical way to advance a whole month and land on
			// the same week offset and weekday.

			// Calculate the next month date.
			//echo "Weekday is: ".$wday."<br>";
			//if ($dateinfo['wday'] > $wday) {
				// The month starts on a week day that is after the target week day.
				// For more detailed discussion of the following formula, see the
				// analysis docs, sdk/analysis/subsystems/datetime.txt

				// TARGET_WDAY is $wday
				// START_WDAY is $startmonthinfo['wday']
				//eDebug($dateinfo);
				//echo 'mdate = $wday - $dateinfo[\'wday\'] + ( 7 * $week ) + 1;<br>';
				//echo "mdate = ".$wday." - ".$dateinfo['wday']." + ( 7 * ".$week." ) + 1<br>";
				//$mdate = $wday - $dateinfo['wday'] + ( 7 * $week ) + 1;
				//echo "mdate: ".$mdate."<br>";
			//} else {
				// The month starts on a week day that is before or equal to the
				// target week day.  This formula is identical to the one above,
				// except that we subtract one from the week offset
				// For more detailed discussion of the following formula, see the
				// analysis docs, sdk/analysis/subsystems/datetime.txt

				// TARGET_WDAY is $wday
				// START_WDAY is $startmonthinfo['wday']
				//$mdate = $wday - $dateinfo['wday'] + ( 7 * ( $week - 1 ) ) + 1;
			//}

		}

		// Re-assemble the $curdate value, using the correct $mdate.  If not doing by_day
		// recurrence, this value remains essentially unchanged.  Otherwise, it will be
		// set to reflect the new day of the Nth weekday.
		//echo "month: ".$dateinfo['mon']."<br>";
		//echo "mdate: ".$mdate."<br>";
		$curdate = exponent_datetime_startOfDayTimestamp(mktime(8,0,0,$dateinfo['mon'],$mdate,$dateinfo['year']));
	} while ($curdate <= $end);

	//exit();
	return $dates;
}

/* exdoc
 * Finds all of the dates that fall within the yearly recurrence criteria
 * (similar to monthly) and within the $start to $end timestamp range.
 * Unlike monthly recurrence, yearly cannot do recurrence like 'the
 * 17th sunday of the year'.
 *
 * @param timestamp $start The start of the recurrence range
 * @param timestamp $end The end of the recurrence range
 * @param integer $freq Yearly frequency - 1 means every year, 2 means every
 *   other year, etc.
 * @node Subsystems:DateTime
 */
function exponent_datetime_recurringYearlyDates($start,$end,$freq) {
	$dates = array();

	$freq = '+'.$freq.' year';
	while ($start <= $end) {
		$dates[] = $start;
		$start = strtotime($freq,$start);
	}

	return $dates;
}

/* exdoc
 * Adapted from calendar module's minical view to be more modular.
 *
 */

function exponent_datetime_monthlyDaysTimestamp($timestamp) {
	global $db;
	$monthly = array();
	$info = getdate($timestamp);
	$week = 0;

	$infofirst = getdate(mktime(12,0,0,$info['mon'],1,$info['year']));
	$weekday = $infofirst['wday']; // day number in grid.  if 7+, switch weeks

	// Grab non-day numbers only (before end of month)
	if ($weekday == (0+ DISPLAY_START_OF_WEEK)) $monthly[$week] = array(); // initialize for non days
	$days_before = 0 - $weekday + DISPLAY_START_OF_WEEK;
	if (($weekday == 0) && (DISPLAY_START_OF_WEEK == 1)) {
		$days_before = -6;
		$weekday = 7;
	}
	for ($i = $days_before; $i < 0; $i++) {
		$monthly[0][$i] = array("ts"=>-1);
	}

	$endofmonth = date('t', $timestamp);
	for ($i = 1; $i <= $endofmonth; $i++) {
		$start = mktime(0,0,0,$info['mon'],$i,$info['year']);
		if ($i == $info['mday']) $currentweek = $week;

		$monthly[$week][$i] = array("ts"=>$start);
		if ($weekday >= (6 + DISPLAY_START_OF_WEEK)) {
			$week++;
			$monthly[$week] = array(); // allocate an array for the next week
			$weekday = 0 + DISPLAY_START_OF_WEEK;
		} else $weekday++;
	}

	// Grab non-day numbers only (after end of month)
	if ($weekday != DISPLAY_START_OF_WEEK) {
		for ($i = 1; $weekday && $i <= (7-$weekday)+DISPLAY_START_OF_WEEK; $i++) $monthly[$week][$i+$endofmonth] = -1;
	}
	return $monthly;
}
?>
